<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Klarna
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Klarna\Extension;

use Joomla\CMS\Log\Log;
use Joomla\Event\Event;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Application\CMSApplication;
use JoomShaper\Plugin\EasyStore\Klarna\Utils\KlarnaApi;
use JoomShaper\Plugin\EasyStore\Klarna\Utils\KlarnaConstants;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class KlarnaPayment extends PaymentGatewayPlugin
{
    /** @var CMSApplication */
    protected $app;

    /**
     * Handles payment events.
     *
     * @param Event $event -- The event object that contains information about the payment.
     * @since 1.0.0
     */
    public function onPayment(Event $event)
    {
        $arguments      = $event->getArguments();
        $paymentData    = $arguments['subject'] ?: new \stdClass();
        $totalPrice     = $paymentData->total_price_in_smallest_unit;
        $tax            = $paymentData->tax_in_smallest_unit ?? 0;
        $shipping       = $paymentData->shipping_charge_in_smallest_unit ?? 0;
        $coupon         = -$paymentData->coupon_discount_amount_in_smallest_unit ?? 0;
        $names          = $this->extractNames($paymentData->user_name);

        $order = (object) [
            'firstname' => $names->first_name,
            'lastname' => $names->last_name,
            'email' => $paymentData->user_email,
            'order_amount' => $totalPrice,
            'purchase_currency' => $paymentData->currency ?? 'GBP',
            'purchase_country' => $paymentData->country->iso2 ?? 'GB'
        ];

        foreach ($paymentData->items as $item) {
            $product_price = is_null($item->discounted_price) ? $item->regular_price_in_smallest_unit : $item->discounted_price_in_smallest_unit;

            $order->items[] = (object) [
                'product' => (object) [
                    'reference'     => $item->id,
                    'name'          => $item->title,
                    'quantity'      => $item->quantity,
                    'product_price' => $product_price,
                ]
            ];
        }

        // Set Tax info
        $order->items[] = (object) [
            'product' => (object) [
                'reference' => 'tax_ref',
                'name' => 'Tax',
                'quantity' => 1,
                'product_price' => $tax,
            ]
        ];

        // Set Shipping info
        $order->items[] = (object) [
            'product' => (object) [
                'reference' => 'ship_ref',
                'name' => $paymentData->shipping_method,
                'quantity' => 1,
                'product_price' => $shipping,
            ]
        ];

        if (!empty($paymentData->coupon_discount_amount)) {
            // Set Discount info
            $order->items[] = (object) [
                'product' => (object) [
                    'reference' => 'discount_ref',
                    'name' => 'Coupon Discount',
                    'quantity' => 1,
                    'product_price' => $coupon,
                ]
            ];
        }

        $klarna = new KlarnaApi();
        $sessionId = $klarna->createSession($order);
        $orderId = $paymentData->order_id;

        if ($sessionId) {
            if (!$klarna->createHPPSession($sessionId, $orderId)) {
                $this->app->enqueueMessage(Text::_('COM_EASYSTORE_KLARNA_ERROR'), 'error');
                $this->app->redirect($paymentData->back_to_checkout_page);
            }
        } else {
            $this->app->enqueueMessage(Text::_('COM_EASYSTORE_KLARNA_ERROR'), 'error');
            $this->app->redirect($paymentData->back_to_checkout_page);
        }
    }

    /**
     * Handles payment notifications.
     *
     * @param Event $event -- The event object that contains information about the payment.
     * @since 1.0.0
     */
    public function onPaymentNotify(Event $event)
    {
        $arguments         = $event->getArguments();
        $paymentNotifyData = $arguments['subject'] ?: new \stdClass();

        try {
            $payload = json_decode($paymentNotifyData->raw_payload);
            $orderId = $paymentNotifyData->get_data['order_id'];

            http_response_code(200);

            if ($payload->session->status === 'WAITING' || $payload->session->status === 'BACK' || $payload->session->status === 'IN_PROGRESS' || $payload->session->status === 'MANUAL_ID_CHECK' || $payload->session->status === 'COMPLETED' || $payload->session->status === 'CANCELLED' || $payload->session->status === 'FAILED' || $payload->session->status === 'DISABLED' || $payload->session->status === 'ERROR') {
                $paymentStatus = 'unpaid';

                if ($payload->session->status === 'COMPLETED') {
                    $paymentStatus = 'paid';
                } elseif ($payload->session->status === 'WAITING' || $payload->session->status === 'IN_PROGRESS' || $payload->session->status === 'MANUAL_ID_CHECK') {
                    $paymentStatus = 'pending';
                } elseif ($payload->session->status === 'BACK' || $payload->session->status === 'CANCELLED' || $payload->session->status === 'FAILED' || $payload->session->status === 'DISABLED' || $payload->session->status === 'ERROR') {
                    $paymentStatus = 'failed';
                }

                $data = (object) [
                    'id'                   => $orderId,
                    'payment_status'       => $paymentStatus,
                    'payment_error_reason' => null,
                    'transaction_id'       => $payload->session->session_id,
                    'payment_method'       => $this->_name
                ];

                $paymentNotifyData->order->updateOrder($data);
            }
        } catch (\Exception $e) {
            Log::add($e->getMessage(), Log::ERROR, 'klarna.easystore');
            http_response_code(404);
        }
    }

    /**
     * Checks if the Klarna username & password is set before processing a payment.
     *
     * @return void -- Returns true if username & password is not empty, otherwise, returns false.
     * @since 1.0.0
     */
    public function onBeforePayment(Event $event)
    {
        $username               = (new KlarnaConstants())->getUsername();
        $password               = (new KlarnaConstants())->getPassword();
        $isRequiredFieldsFilled = false;

        if (!empty($username) && !empty($password)) {
            $isRequiredFieldsFilled = true;
        }

        $event->setArgument('result', $isRequiredFieldsFilled);
    }

    /**
     * Function to extract names
     *
     * @param string $fullName
     * @return object
     */
    private function extractNames($fullName)
    {
        // Split the full name into an array of words
        $nameParts = explode(' ', $fullName);

        // Extract the first name (concatenate the first part(s))
        $firstName = implode(' ', array_slice($nameParts, 0, -1));

        // Extract the last name (use the last part)
        $lastName = end($nameParts);

        return (object) ['first_name' => $firstName, 'last_name' => $lastName];
    }
}
