<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.klarna
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Klarna\Utils;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use JoomShaper\Plugin\EasyStore\Klarna\Utils\KlarnaConstants;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

class KlarnaApi
{
    /**
     * Creates a session to initiate SessionId
     *
     * @param object $order
     * @return string|bool
     */
    public function createSession($order)
    {
        $constants = new KlarnaConstants();
        $klarnaUrl = "{$constants->getKlarnaBaseUrl()}/payments/v1/sessions";

        // Build order_lines
        $contents = $order->items;
        foreach ($contents as $line) {
            $product_price = $line->product->product_price;

            $order_line = array(
                "type" => "physical",
                "reference" => $line->product->reference,
                "name" => $line->product->name,
                "quantity" => $line->product->quantity,
                "unit_price" => $product_price,
                "total_amount" => $product_price * $line->product->quantity,
            );

            $order_lines[] = $order_line;
        }

        $data = array(
            "intent" => "buy",
            "purchase_country" => $order->purchase_country,
            "purchase_currency" => $order->purchase_currency,
            "order_amount" => $order->order_amount,
            "order_lines" => $order_lines,
        );

        $payload = json_encode($data);
        $header = [
            "Content-Type: application/json",
            "Cache-Control: no-cache",
            "Authorization: Basic " . base64_encode("{$constants->getUsername()}:{$constants->getPassword()}")
        ];

        try {
            $response = $this->httpPost($klarnaUrl, $payload, $header);
            $result = json_decode($response, true);

            if (!isset($result['session_id'])) {
                Log::add($response, Log::ERROR, 'klarna.easystore');
            }

            return isset($result['session_id']) ? $result['session_id'] : false;
        } catch (\Exception $e) {
            // Handle the exception, log it, or rethrow if needed
            Log::add($e->getMessage(), Log::ERROR, 'klarna.easystore');

            return false;
        }
    }

    /**
     * Create the HPP Session and redirect to Klarna
     *
     * @param string $sessionId
     * @param int $orderId
     * @return void
     */
    public function createHPPSession($sessionId, $orderId)
    {
        $constants = new KlarnaConstants();
        $hppUrl = "{$constants->getKlarnaBaseUrl()}/hpp/v1/sessions";

        $data = array(
            "payment_session_url" => "{$constants->getKlarnaBaseUrl()}/payments/v1/sessions/$sessionId",
            "merchant_urls" => array(
                "success" => $constants->getSuccessUrl(),
                "cancel" => $constants->getCancelUrl($orderId),
                "back" => $constants->getCancelUrl($orderId),
                "status_update" => $constants->getWebhookUrl() . '&order_id=' . $orderId,
            ),
            "options" => array(
                "place_order_mode" => "CAPTURE_ORDER",
            ),
        );

        $payload = json_encode($data);
        $header = [
            "Content-Type: application/json",
            "Cache-Control: no-cache",
            "Authorization: Basic " . base64_encode("{$constants->getUsername()}:{$constants->getPassword()}")
        ];

        try {
            $response = $this->httpPost($hppUrl, $payload, $header);
            $result = json_decode($response, true);

            if ($response === false) {
                throw new \Exception("HTTP request failed.");
            }

            if (isset($result['redirect_url'])) {
                Factory::getApplication()->redirect($result['redirect_url']);
                die;
            }
        } catch (\Exception $e) {
            // Handle the exception, log it, or rethrow if needed
            Log::add($e->getMessage(), Log::ERROR, 'klarna.easystore');

            return false;
        }
    }

    /**
     * Request an Http Post request
     *
     * @param string $url
     * @param mixed $data
     * @param array $header
     * @return string
     */
    private function httpPost($url, $data, $header)
    {
        $ch = curl_init($url);

        curl_setopt($ch, CURLOPT_HTTPHEADER, $header);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data);

        $response = curl_exec($ch);
        curl_close($ch);

        if ($response === false) {
            Log::add("cURL error: " . curl_error($ch) . " (Code: " . curl_errno($ch) . ")", Log::ERROR, 'klarna.easystore');
            http_response_code(404);

            throw new \Exception("cURL error: " . curl_error($ch) . " (Code: " . curl_errno($ch) . ")");
        }

        return $response;
    }
}
