<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Stripe
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Stripe\Utils;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Stripe\StripeClient;
use Stripe\Checkout\Session;
use Joomla\CMS\Language\Text;
use Stripe\Exception\ApiErrorException;
use Joomla\CMS\Application\CMSApplication;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Class that contains Api for the Stripe payment gateway.
 * @since 1.0.0
*/
class StripeApi 
{
    /** @var CMSApplication */
    protected static $app;

    /**
     * Get the checkout URL for Stripe.
     *
     * @param array  $lineItems      Line items for the checkout session.
     * @param object $productsList   The list of products and additional information.
     *
     * @return array               The checkout session URL.
     * @since  1.0.0
     */
    public static function getCheckoutUrl($lineItems,$paymentData) {

        $couponId        = null;
        $constants       = new StripeConstants();
        $shippingOptions = StripeHelper::getShippingOptions($paymentData);

        if (!is_null($paymentData->coupon_discount_amount)) 
        {   
            $couponDiscountAmount = number_format($paymentData->coupon_discount_amount,2,$paymentData->decimal_separator,'');
            $coupon   = self::createCoupon($couponDiscountAmount,$paymentData->currency);
            $couponId = $coupon->id ?: null;
        }

        $checkout_session_data = 
        [
            'line_items'    => $lineItems, 
            'metadata'      => [
                'payment_type' => 'stripe', 
                'order_id'     => $paymentData->order_id , 
                'coupon_id'    => $couponId
            ],
            'mode'          => 'payment',
            'success_url'   => $constants->getSuccessUrl(),
            'cancel_url'    => $constants->getCancelUrl(),
            'payment_intent_data' => [
                'metadata' => ['order_id' => $paymentData->order_id,'coupon_id' => $couponId],
            ],
            'shipping_options' => $shippingOptions,
        ];

        if (!is_null($couponId)) 
        {
            $checkout_session_data['discounts'] = [
                [
                    'coupon' => $couponId,
                ],
            ];
        }

        try {
           $checkout_session = Session::create($checkout_session_data);
           return ['response' => 'success', 'message' => $checkout_session->url];
        } catch (\Throwable $th) {
            Log::add($th->getMessage(), Log::ERROR, 'stripe.easystore');
            return ['response' => 'error', 'message' => $th->getMessage()];
        } 
    }


    /**
     * Create a coupon with a specific discount amount and currency.
     *
     * @param float  $couponAmount The discount amount to apply.
     * @param string $currency     The currency code.
     *
     * @return \Stripe\Coupon|null The created coupon object or null if an error occurred.
     * @since  1.0.0
     */
    public static function createCoupon($couponAmount,$currency)
    {       
        self::$app       = Factory::getApplication();
        $constants       = new StripeConstants();
        $stripeSecretKey = $constants->getSecretKey();
        $stripe          = new StripeClient($stripeSecretKey);

        if (empty($stripeSecretKey)) 
        {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_SECRET_KEY_EMPTY_FIELD'), Log::ERROR, 'stripe.easystore');
            return null;
        }

        try {
            $coupon = $stripe->coupons->create([
                        'amount_off' => (float) $couponAmount * 100,
                        'duration'   => 'once',
                        'currency'   => strtolower($currency)
                    ]); 

            return $coupon;
        } catch (\Throwable $e) {
            Log::add($e->getMessage(), Log::ERROR,'stripe.easystore');
            self::$app->enqueueMessage($e->getMessage(),'error');
        }
    }

    /**
     * Create a new webhook URL in Stripe with the given secret key.
     *
     * @param  string $stripeSecretKey - The Stripe secret key to use for creating the webhook.
     *
     * @return mixed                   - The newly created webhook object or an error message if an error occurs.
     * @since  1.0.0
     */
    public static function createNewWebhookUrl($stripeSecretKey){
        
        self::$app  = Factory::getApplication();
        $stripe     = new StripeClient($stripeSecretKey);
        $constant   = new StripeConstants();
        $webhookUrl = $constant->getWebHookUrl();

        try 
        {
            $newCreatedWebhook = $stripe->webhookEndpoints->create([
                'url'            => $webhookUrl,
                'enabled_events' => [
                    'payment_intent.payment_failed',
                    'checkout.session.completed',
                    'payment_intent.canceled'
                ],
            ]);

            return $newCreatedWebhook;
        } catch (ApiErrorException $error) 
        {
            Log::add($error->getMessage(), Log::ERROR,'stripe.easystore');
            self::$app->enqueueMessage($error->getMessage(), 'error');
        }
    }

    /**
     * Delete a coupon with the specified ID from Stripe.
     *
     * @param string $couponID -- The ID of the coupon to be deleted.
     * @since 1.0.0
     */
    public static function deleteCoupon($couponID)
    {
        self::$app = Factory::getApplication();
        $constants = new StripeConstants();
        $secretKey = $constants->getSecretKey();
        $stripe    = new StripeClient($secretKey);

        if (empty($secretKey)) {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_SECRET_KEY_EMPTY_FIELD'), Log::ERROR, 'stripe.easystore');
            return false;
        }

        try {
            $stripe->coupons->delete($couponID, []);
        } catch (ApiErrorException $error) {
            Log::add($error->getMessage(), Log::ERROR,'stripe.easystore');
            self::$app->enqueueMessage($error->getMessage(),'error');
        }
    }
} 