<?php
/**
 * @package     Joomla.Administrator
 * @subpackage  Search
 *
 * @copyright   (C) 2005 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

use Joomla\CMS\Language\Multilanguage;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\Component\Tags\Site\Helper\RouteHelper;
use Joomla\Database\ParameterType;

/**
 * Tags search plugin.
 *
 * @since  3.3
 */
class PlgSearchTags extends CMSPlugin
{
	/**
	 * Application object
	 *
	 * @var    \Joomla\CMS\Application\CMSApplicationInterface
	 * @since  4.0.0
	 */
	protected $app;

	/**
	 * Database Driver Instance
	 *
	 * @var    \Joomla\Database\DatabaseDriver
	 * @since  4.0.0
	 */
	protected $db;

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  3.3
	 */
	protected $autoloadLanguage = true;

	/**
	 * Determine areas searchable by this plugin.
	 *
	 * @return  array  An array of search areas.
	 *
	 * @since   3.3
	 */
	public function onContentSearchAreas()
	{
		static $areas = array(
			'tags' => 'PLG_SEARCH_TAGS_TAGS'
		);

		return $areas;
	}

	/**
	 * Search content (tags).
	 *
	 * The SQL must return the following fields that are used in a common display
	 * routine: href, title, section, created, text, browsernav.
	 *
	 * @param   string  $text      Target search string.
	 * @param   string  $phrase    Matching option (possible values: exact|any|all).  Default is "any".
	 * @param   string  $ordering  Ordering option (possible values: newest|oldest|popular|alpha|category).  Default is "newest".
	 * @param   string  $areas     An array if the search is to be restricted to areas or null to search all areas.
	 *
	 * @return  array  Search results.
	 *
	 * @since   3.3
	 */
	public function onContentSearch($text, $phrase = '', $ordering = '', $areas = null)
	{
		$db    = $this->db;
		$query = $db->getQuery(true);
		$app   = $this->app;
		$user  = $app->getIdentity();
		$lang  = $app->getLanguage();

		$section = Text::_('PLG_SEARCH_TAGS_TAGS');
		$limit   = $this->params->def('search_limit', 50);

		if (is_array($areas) && !array_intersect($areas, array_keys($this->onContentSearchAreas())))
		{
			return array();
		}

		$text = trim($text);

		if ($text === '')
		{
			return array();
		}

		$text = '%' . $text . '%';

		switch ($ordering)
		{
			case 'alpha':
				$order = 'a.title ASC';
				break;

			case 'newest':
				$order = 'a.created_time DESC';
				break;

			case 'oldest':
				$order = 'a.created_time ASC';
				break;

			case 'popular':
			default:
				$order = 'a.title DESC';
		}

		$query->select('a.id, a.title, a.alias, a.note, a.published, a.access'
			. ', a.checked_out, a.checked_out_time, a.created_user_id'
			. ', a.path, a.parent_id, a.level, a.lft, a.rgt'
			. ', a.language, a.created_time AS created, a.description');

		$caseWhen  = ' CASE WHEN ';
		$caseWhen .= $query->charLength('a.alias', '!=', '0');
		$caseWhen .= ' THEN ';
		$a_id                  = $query->castAsChar('a.id');
		$caseWhen .= $query->concatenate(array($a_id, 'a.alias'), ':');
		$caseWhen .= ' ELSE ';
		$caseWhen .= $a_id . ' END as slug';
		$query->select($caseWhen);

		$query->from('#__tags AS a');
		$query->where('a.alias <> ' . $db->quote('root'));

		$query->where('(a.title LIKE :title OR a.alias LIKE :alias)')
			->bind(':title', $text)
			->bind(':alias', $text);
		
		$query->where($db->quoteName('a.published') . ' = 1');

		if (!$user->authorise('core.admin'))
		{
			$query->whereIn($db->quoteName('a.access'), $user->getAuthorisedViewLevels());
		}

		if ($app->isClient('site') && Multilanguage::isEnabled())
		{
			$query->whereIn($db->quoteName('a.language'), [$app->getLanguage()->getTag(), '*'], ParameterType::STRING);
		}

		$query->order($order);

		$db->setQuery($query, 0, $limit);

		try
		{
			$rows = $db->loadObjectList();
		}
		catch (RuntimeException $e)
		{
			$rows = array();
			$app->enqueueMessage(Text::_('JERROR_AN_ERROR_HAS_OCCURRED'), 'error');
		}

		if ($rows)
		{
			foreach ($rows as $key => $row)
			{
				$rows[$key]->href       = RouteHelper::getTagRoute($row->slug);
				$rows[$key]->text       = ($row->description !== '' ? $row->description : $row->title);
				$rows[$key]->text      .= $row->note;
				$rows[$key]->section    = $section;
				$rows[$key]->created    = $row->created;
				$rows[$key]->browsernav = 0;
			}
		}

		if (!$this->params->get('show_tagged_items', 0))
		{
			return $rows;
		}
		else
		{
			$final_items = $rows;
			$tag_model   = $app->bootComponent('com_tags')
				->getMVCFactory()->createModel('Tag', 'Site');
			$tag_model->getState();

			foreach ($rows as $key => $row)
			{
				$tag_model->setState('tag.id', $row->id);
				$tagged_items = $tag_model->getItems();

				if ($tagged_items)
				{
					foreach ($tagged_items as $k => $item)
					{
						// For 3rd party extensions we need to load the component strings from its sys.ini file
						$parts = explode('.', $item->type_alias);
						$comp = array_shift($parts);
						$lang->load($comp, JPATH_SITE, null, false, true)
						|| $lang->load($comp, JPATH_SITE . '/components/' . $comp, null, false, true);

						// Making up the type string
						$type = implode('_', $parts);
						$type = $comp . '_CONTENT_TYPE_' . $type;

						$new_item        = new stdClass;
						$new_item->href  = $item->link;
						$new_item->title = $item->core_title;
						$new_item->text  = $item->core_body;

						if ($lang->hasKey($type))
						{
							$new_item->section = Text::sprintf('PLG_SEARCH_TAGS_ITEM_TAGGED_WITH', Text::_($type), $row->title);
						}
						else
						{
							$new_item->section = Text::sprintf('PLG_SEARCH_TAGS_ITEM_TAGGED_WITH', $item->content_type_title, $row->title);
						}

						$new_item->created    = $item->displayDate;
						$new_item->browsernav = 0;
						$final_items[]        = $new_item;
					}
				}
			}

			return $final_items;
		}
	}
}
