<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.Stripe
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Stripe\Utils;

use Joomla\Registry\Registry;
use Joomla\CMS\Response\JsonResponse;
use JoomShaper\Component\EasyStore\Administrator\Plugin\Constants;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Class that contains constants for the Stripe payment gateway.
 * @since 1.0.0
 */
class StripeConstants extends Constants
{
    /**
     * Plugin parameters
     *
     * @var Registry
     */
    protected $params;

    /**
     * The payment plugin name
     *
     * @var string
     */
    protected $name = 'stripe';

    /**
     * The constructor method
     */
    public function __construct()
    {
        parent::__construct($this->name);
    }

    /**
     * Get the webhook secret key
     *
     * @return string
     * @since  1.0.0
     */
    public function getWebHookSecretKey()
    {
        return $this->getPaymentEnvironment() === 'test' ? self::getTestWebhookSecretKey() : self::getLiveWebhookSecretKey();
    }

    /**
     * Sends a JSON response with the webhook secret key and a corresponding message.
     *
     * If the webhook secret key is provided, it indicates that a new webhook endpoint was created.
     * Otherwise, it signifies that the webhook endpoint already exists.
     *
     * @param  string       $message          -- Display Message
     * @param  string|null  $webhookSecretKey -- The webhook secret key (optional).
     * @return void
     * @since  1.0.5
     */

    public function sendResponse($message, $webhookSecretKey = null)
    {
        $this->app->setHeader('Content-Type', 'application/json');
        $this->app->setHeader('status', 200, true);
        $this->app->sendHeaders();

        echo new JsonResponse(['webhook_secret' => $webhookSecretKey,'message' => $message]);

        $this->app->close();
    }


    /**
     * Get Test Secret Key
     *
     * @return string
     * @since  1.0.6
     */

    public function getTestKey()
    {
        return $this->params->get('stripe_secret_key', '');
    }

    public function getTestWebhookSecretKey()
    {
        return $this->params->get('stripe_webhook_secret_key', '');
    }

    public function getLiveWebhookSecretKey()
    {
        return $this->params->get('stripe_webhook_live_secret_key', '');
    }

    /**
     * Get Secret  key based on payment environment
     *
     * @return string
     * @since  1.0.6
     */

    public function getSecretKey($paymentEnvironment = null): string
    {
        $environment = !empty($paymentEnvironment) ? $paymentEnvironment : $this->getPaymentEnvironment();
        return  $environment === 'test' ? self::getTestKey() : $this->getLiveKey();
    }
}
