<?php

/**
 * @package     Joomla.Administrator
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 - 2024 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Administrator\Helper;

use Throwable;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\Registry\Registry;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\Database\DatabaseInterface;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Get settings data
 *
 * @since 1.0.0
 */
final class SettingsHelper
{
    /**
     * Get settings value by using key and value
     *
     * $settings = SettingsHelper::getSettings();
     *
     * $currency = $settings->get('general.currency', 'USD:$');
     *
     * @return object
     *
     * @since 1.0.0
     */
    public static function getSettings()
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $query->select($db->quoteName(['key', 'value']))->from($db->quoteName('#__easystore_settings'));
        $db->setQuery($query);

        try {
            $settings = $db->loadObjectList('key') ?? [];

            foreach ($settings as &$setting) {
                $setting = json_decode($setting->value ?? '') ?? null;
            }

            unset($setting);

            $settings = (object) $settings;

            return new Registry($settings);
        } catch (Throwable $error) {
            return new Registry([]);
        }
    }

    /**
     * Function to set Settings data by key value pair
     *
     * @param string $key   The key of the settings to update ie 'general.storeName'
     * @param mixed $value  Value of the key
     * @return object
     */
    public static function setSettings(string $key, mixed $value)
    {
        $db    = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true);
        $query->select($db->quoteName(['key', 'value']))->from($db->quoteName('#__easystore_settings'));
        $db->setQuery($query);

        try {
            $response = new \stdClass();
            $settings = $db->loadObjectList('key') ?? [];

            foreach ($settings as &$setting) {
                $setting = json_decode($setting->value ?? '') ?? null;
            }

            unset($setting);

            $settings = (object) $settings;
            $keys     = explode('.', $key);
            $temp     = &$settings;

            try {
                foreach ($keys as $key) {
                    if (!isset($temp->$key)) {
                        $temp->$key = new \stdClass();
                    }
                    $temp = &$temp->$key;
                }

                $temp = $value;
            } catch (\Exception $error) {
                $response->status  = false;
                $response->message = $error->getMessage();

                return $response;
            }

            // Update the settings data
            foreach ($settings as $key => $value) {
                if ($keys[0] == $key) {
                    $settingsData        = new \stdClass();
                    $settingsData->key   = $key;
                    $settingsData->value = json_encode($value ?? '');

                    EasyStoreDatabaseOrm::updateOrCreate('#__easystore_settings', $settingsData, 'key');
                }
            }

            $response->status  = true;
            $response->message = 'settings updated';

            return $response;
        } catch (\Exception $error) {
            $response->status  = false;
            $response->message = $error->getMessage();

            return $response;
        }
    }

    /**
     * Get the store address details.
     *
     * @return array
     *
     * @since 1.0.2
     * @since 1.2.0 add email key
     */
    public static function getAddress()
    {
        $settings = self::getSettings();

        $countryId = $settings->get('general.country', '');
        $stateId   = $settings->get('general.state', '');

        $CountryCityNames = EasyStoreHelper::getCountryStateFromJson($countryId, $stateId);

        return [
            'address_1' => $settings->get('general.addressLineOne', ''),
            'address_2' => $settings->get('general.addressLineTwo', ''),
            'city'      => $settings->get('general.city', ''),
            'state'     => $CountryCityNames->state,
            'zip_code'  => $settings->get('general.postcode', ''),
            'country'   => $CountryCityNames->country,
            'name'      => $settings->get('general.storeName', ''),
            'email'     => $settings->get('general.storeEmail', ''),
            'phone'     => $settings->get('general.storePhone', ''),
        ];
    }

    /**
     * Get Payment plugin list
     *
     * @return mixed
     */
    public static function getPluginSchema()
    {
        $cachePath = JPATH_CACHE . '/easystore';
        $cacheFile = $cachePath . '/payments.json';

        $url     = 'https://www.joomshaper.com/products/easystore/payments.json';
        $content = '';

        if (!file_exists($cachePath)) {
            Folder::create($cachePath, 0755);
        }

        if (file_exists($cacheFile) && (filemtime($cacheFile) > (time() - (24 * 60 * 60)))) {
            if (ini_get('allow_url_fopen')) {
                $content = file_get_contents($cacheFile);
            } else {
                $handle  = fopen($cacheFile, "r");
                $content = fread($handle, filesize($cacheFile));
                fclose($handle);
            }
        } else {
            if (ini_get('allow_url_fopen')) {
                $content = file_get_contents($url);
            } else {
                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, $url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);

                $content = curl_exec($ch);
                curl_close($ch);
            }

            if (!empty($content)) {
                File::write($cacheFile, $content);
            }
        }

        if (empty($content)) {
            return [];
        }

        return json_decode($content);
    }

    public static function getWeightWithUnit($weight)
    {
        $settings = self::getSettings();
        $standardUnits = $settings->get('products.standardUnits.weight', 'kg');
    
        // Corrected key from 'og' to 'oz'
        $unitLanguageString = [
            'g'  => 'COM_EASYSTORE_UNIT_VALUE_FOR_GRAM',
            'kg' => 'COM_EASYSTORE_UNIT_VALUE_FOR_KILOGRAM',
            'lb' => 'COM_EASYSTORE_UNIT_VALUE_FOR_POUND',
            'oz' => 'COM_EASYSTORE_UNIT_VALUE_FOR_OUNCE', // Corrected key
        ];
    
        // Check if the key exists in the array to avoid undefined key warnings
        $weightUnit = isset($unitLanguageString[$standardUnits]) ? Text::_($unitLanguageString[$standardUnits]) : '';
    
        return $weight . $weightUnit;
    }

    /**
     * Checks if a specific email template type is enabled.
     *
     * This function retrieves the settings and checks if the specified email template
     * type (e.g., "order_confirmation") is enabled.
     *
     * @param string $templateType The type of email template to check (e.g., "order_confirmation").
     * @return bool Returns true if the email template type is enabled, false otherwise.
     *
     * @since 1.2.0
     */
    public static function isEmailTemplateEnabled($templateType)
    {
        $settings       = self::getSettings();
        $emailTemplates = $settings->get('email_templates', '');
        $isEmailEnabled = false;

        $key = explode('_', $templateType)[0];

        // Check if email templates are enabled
        if (!empty($emailTemplates) && is_array($emailTemplates->$key->templates)) {
            foreach ($emailTemplates->$key->templates as $template) {
                if (!empty($template->type) && $template->type === $templateType) {
                    $isEmailEnabled = $template->is_enabled;
                    break; // Exit loop once the relevant template is found
                }
            }
        }

        return $isEmailEnabled;
    }

    /**
     * Checks if guest checkout is enabled.
     *
     * This function retrieves the application settings and user identity to determine
     * if guest checkout is allowed. It returns true if the current user is a guest
     * and guest checkout is enabled in the settings, false otherwise.
     *
     * @return bool Returns true if guest checkout is enabled for the current user, false otherwise.
     *
     * @since 1.2.0
     */
    public static function isGuestCheckoutEnable()
    {
        $app                = Factory::getApplication();
        $user               = $app->getIdentity();

        $settings           = self::getSettings();
        $allowGuestCheckout = $settings->get('checkout.allow_guest_checkout', false);

        return ($user->guest && $allowGuestCheckout);
    }
}
