<?php

/**
 * @package     EasyStore.Site
 * @subpackage  EasyStore.stripe
 *
 * @copyright   Copyright (C) 2023 - 2024 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Plugin\EasyStore\Stripe\Extension;

require_once __DIR__ . '/../../vendor/autoload.php';

use Stripe\Stripe;
use Stripe\Webhook;
use Joomla\CMS\Log\Log;
use Joomla\Event\Event;
use Stripe\StripeClient;
use Joomla\CMS\Language\Text;
use Stripe\Event as StripeEvent;
use Joomla\Event\SubscriberInterface;
use Stripe\Exception\ApiErrorException;
use Joomla\CMS\Application\CMSApplication;
use Stripe\Exception\SignatureVerificationException;
use JoomShaper\Plugin\EasyStore\Stripe\Utils\StripeApi;
use JoomShaper\Component\EasyStore\Site\Helper\ArrayHelper;
use JoomShaper\Plugin\EasyStore\Stripe\Utils\StripeConstants;
use JoomShaper\Component\EasyStore\Administrator\Plugin\PaymentGatewayPlugin;

/**
 * StripePayment class.
 *
 * This class extends the PaymentGatewayPlugin class and provides implementation for integrating the Stripe
 * payment gateway.
 *
 * It handles various payment-related functionalities such as initiating payments, handling successful
 * payments and managing canceled payments.
 *
 * @since 1.0.0
 */
class StripePayment extends PaymentGatewayPlugin implements SubscriberInterface
{
    /** @var CMSApplication */
    protected $app;

    /**
     * Initiate an event that will lead to a redirection to the Stripe checkout page.
     *
     * @param Event|string  $event  The event object.
     * @since 1.0.0
     */
    public function onPayment(Event $event)
    {
        $lineItems   = [];
        $constants   = new StripeConstants();
        $secretKey   = $constants->getSecretKey();
        $arguments   = $event->getArguments();
        $paymentData = $arguments['subject'] ?: new \stdClass();

        if (empty($secretKey)) {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_SECRET_KEY_EMPTY_FIELD'), Log::ERROR, 'stripe.easystore');
            $this->app->enqueueMessage(Text::_('COM_EASYSTORE_STRIPE_ERROR') . ' ' . Text::_('COM_EASYSTORE_STRIPE_SECRET_KEY_EMPTY_FIELD'), 'error');
            $this->app->redirect($paymentData->back_to_checkout_page);
        }

        Stripe::setApiKey($secretKey);
        header('Content-Type: application/json');

        if (!empty($paymentData)) {
            // Create line items for products
            foreach ($paymentData->items as $product) {
                $price = !is_null($product->discounted_price) ? $product->discounted_price_in_smallest_unit : $product->regular_price_in_smallest_unit;

                $lineItems [] = [
                    'price_data' => [
                        'product_data' => [
                            'name' => $product->title,
                        ],
                        'unit_amount' => (float) $price,
                        'currency'    => strtolower($paymentData->currency),
                    ],
                    'quantity' => $product->quantity,
                ];
            }

            // Create a line item for tax
            if (!is_null($paymentData->tax)) {
                $lineItems[] = [
                    'price_data' => [
                        'product_data' => [
                            'name' => 'Tax',
                        ],
                        'unit_amount' => (float)$paymentData->tax_in_smallest_unit,
                        'currency'    => $paymentData->currency,
                    ],
                    'quantity' => 1,
                ];
            }
        }

        $checkoutSession = StripeApi::getCheckoutUrl($lineItems, $paymentData);

        // Redirect to checkout page
        if ($checkoutSession['response'] === 'success') {
            $this->app->redirect($checkoutSession['message']);
        }

        if ($checkoutSession['response'] === 'error') {
            $this->app->enqueueMessage(Text::_('COM_EASYSTORE_STRIPE_ERROR') . ' ' . Text::_($checkoutSession['message']), 'error');
            $this->app->redirect($paymentData->back_to_checkout_page);
        }
    }

    /**
     * Handle payment notifications from Stripe webhook.
     *
     * @since 1.0.0
     */
    public function onPaymentNotify(Event $event)
    {
        $errorReason       = null;
        $stripeEvent       = null;
        $status            = null;
        $constants         = new StripeConstants();
        $secretKey         = $constants->getSecretKey();
        $arguments         = $event->getArguments();
        $paymentNotifyData = $arguments['subject'] ?: new \stdClass();
        $order             = $paymentNotifyData->order;
        $webhookSecretKey  = $constants->getWebHookSecretKey();
        $payload           = $paymentNotifyData->raw_payload;

        Stripe::setApiKey($secretKey);

        if (empty($webhookSecretKey)) {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_WEBHOOK_SECRET_KEY_EMPTY_FIELD'), Log::ERROR, 'stripe.easystore');
        }

        try {
            $stripeEvent = StripeEvent::constructFrom(json_decode($payload, true));
        } catch (\UnexpectedValueException $error) {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_WEBHOOK_ERROR'), Log::ERROR, 'stripe.easystore');
            echo '⚠️  Webhook error while parsing basic request.';
            http_response_code(400);
            exit();
        }

        if ($webhookSecretKey) {
            $signatureHeader = $paymentNotifyData->server_variables['HTTP_STRIPE_SIGNATURE'];

            try {
                $stripeEvent = Webhook::constructEvent($payload, $signatureHeader, $webhookSecretKey);
                http_response_code(200);
            } catch (SignatureVerificationException $error) {
                Log::add(Text::_('COM_EASYSTORE_STRIPE_INVALID_SIGNATURE'), Log::ERROR, 'stripe.easystore');
                echo '⚠️  Webhook error while validating signature.';
                http_response_code(400);
                exit();
            }
        }

        $paymentDetails = $stripeEvent->data->object;
        $orderID        = $paymentDetails->metadata->order_id;
        $transactionId  = $paymentDetails->payment_intent;
        $couponDiscount = isset($paymentDetails->total_details) ? $paymentDetails->total_details->amount_discount : 0;
        $couponID       = $paymentDetails->metadata->coupon_id;

        // Handle the stripeEvent
        $statusMap = [
            'payment_intent.payment_failed' => 'failed',
            'checkout.session.completed'    => 'paid',
            'payment_intent.canceled'       => 'canceled'
        ];

        if (!empty($stripeEvent->type)) {
            $status = $statusMap[$stripeEvent->type];
        }

        if ($status === 'failed') {
            $errorReason = $paymentDetails->last_payment_error->message;
        }

        // Delete coupon if discount is applied
        if ((float) $couponDiscount > 0 && $couponID) {
            StripeApi::deleteCoupon($couponID);
        }

        $data = (object) [
            'id'                   => $orderID,
            'payment_status'       => $status,
            'payment_error_reason' => $errorReason,
            'transaction_id'       => $transactionId,
            'payment_method'       => $this->_name
        ];

        try {
            $order->updateOrder($data);
        } catch (ApiErrorException $error) {
            Log::add('Stripe Error: ' . $error->getMessage(), Log::ERROR, 'stripe.easystore');
        }

        exit();
    }

    /**
     * Creates a webhook URL for Stripe integration.
     *
     * This function checks if the webhook URL is already added in the Stripe account.
     * If not, it creates a new webhook URL using the provided Stripe secret key.
     * The new webhook URL is then returned in the response.
     *
     * @return void
     */

    public function createWebhook()
    {
        $webhookUrlIsAdded = false;
        $constants         = new StripeConstants();
        $webhookUrl        = $constants->getWebHookUrl();
        $secretKey         = $constants->getSecretKey();

        if (empty($secretKey)) {
            Log::add(Text::_('COM_EASYSTORE_STRIPE_SECRET_KEY_EMPTY_FIELD'), Log::ERROR, 'stripe.easystore');
        }

        try {
            $stripe = new StripeClient($secretKey);
        } catch (\Throwable $error) {
            $constants->sendResponse($error->getMessage());
        }

        $allWebhooks = $stripe->webhookEndpoints->all();

        $webhookUrlIsAdded = ArrayHelper::some(function ($item) use ($webhookUrl) {
            return $item->url === $webhookUrl;
        }, $allWebhooks);

        if (!$webhookUrlIsAdded) {
            $newWebhook    = StripeApi::createNewWebhookUrl($secretKey);

            if (isset($newWebhook->secret)) {
                $constants->sendResponse(Text::_('PLG_EASYSTORE_STRIPE_WEBHOOK_ENDPOINT_CREATED'), $newWebhook->secret);
            }

            if (isset($newWebhook->error) && $newWebhook->error) {
                $constants->sendResponse($newWebhook->message);
            }
        }

        $constants->sendResponse(Text::_('PLG_EASYSTORE_STRIPE_WEBHOOK_ENDPOINT_EXISTS'));
    }

    /**
     * Check if all the required fields for the plugin are filled.
     *
     * @return void The result of the check, indicating whether the required fields are filled.
     * @since  1.0.3
     */
    public function onBeforePayment(Event $event)
    {
        $constants              = new StripeConstants();
        $secretKey              = $constants->getSecretKey();
        $webhookSecretKey       = $constants->getWebHookSecretKey();
        $isRequiredFieldsFilled = !empty($secretKey) && !empty($webhookSecretKey);

        $event->setArgument('result', $isRequiredFieldsFilled);
    }

    /**
     * function for getSubscribedEvents : new Joomla 4 feature
     *
     * @return array
     *
     * @since   1.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAjaxStripe'    => 'createWebhook',
            'onBeforePayment' => 'onBeforePayment',
            'onPayment'       => 'onPayment',
            'onPaymentNotify' => 'onPaymentNotify'
        ];
    }
}
